// Configuration
const CONFIG = {
    // Update this to your backend API URL when deployed
    API_URL: 'http://localhost:8000',
    MAX_RETRIES: 3,
    RETRY_DELAY: 1000
};

// State management
let conversationHistory = [];
let isProcessing = false;

// DOM elements
const chatMessages = document.getElementById('chatMessages');
const messageInput = document.getElementById('messageInput');
const sendButton = document.getElementById('sendButton');
const typingIndicator = document.getElementById('typingIndicator');

// Initialize
document.addEventListener('DOMContentLoaded', () => {
    setupEventListeners();
    checkBackendConnection();
});

// Setup event listeners
function setupEventListeners() {
    sendButton.addEventListener('click', handleSendMessage);
    
    messageInput.addEventListener('keydown', (e) => {
        if (e.key === 'Enter' && !e.shiftKey) {
            e.preventDefault();
            handleSendMessage();
        }
    });

    // Auto-resize textarea
    messageInput.addEventListener('input', () => {
        messageInput.style.height = 'auto';
        messageInput.style.height = messageInput.scrollHeight + 'px';
    });
}

// Check backend connection
async function checkBackendConnection() {
    try {
        const response = await fetch(`${CONFIG.API_URL}/health`);
        if (!response.ok) {
            showError('Backend server is not responding. Please check if the server is running.');
        }
    } catch (error) {
        showError('Cannot connect to backend server. Please ensure the server is running on ' + CONFIG.API_URL);
    }
}

// Handle send message
async function handleSendMessage() {
    const message = messageInput.value.trim();
    
    if (!message || isProcessing) {
        return;
    }

    // Add user message to UI
    addMessage(message, 'user');
    
    // Clear input
    messageInput.value = '';
    messageInput.style.height = 'auto';
    
    // Add to conversation history
    conversationHistory.push({
        role: 'user',
        content: message
    });

    // Show typing indicator
    showTypingIndicator();
    
    // Disable input while processing
    setProcessingState(true);

    try {
        // Send message to backend
        const response = await sendMessageToBackend(message);
        
        // Hide typing indicator
        hideTypingIndicator();
        
        // Add bot response to UI
        addMessage(response.response, 'bot');
        
        // Add to conversation history
        conversationHistory.push({
            role: 'assistant',
            content: response.response
        });
        
    } catch (error) {
        hideTypingIndicator();
        showError('Sorry, I encountered an error processing your request. Please try again.');
        console.error('Error:', error);
    } finally {
        setProcessingState(false);
    }
}

// Send message to backend with retry logic
async function sendMessageToBackend(message, retryCount = 0) {
    try {
        const response = await fetch(`${CONFIG.API_URL}/chat`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                message: message,
                conversation_history: conversationHistory
            })
        });

        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }

        return await response.json();
        
    } catch (error) {
        if (retryCount < CONFIG.MAX_RETRIES) {
            await new Promise(resolve => setTimeout(resolve, CONFIG.RETRY_DELAY));
            return sendMessageToBackend(message, retryCount + 1);
        }
        throw error;
    }
}

// Add message to chat UI
function addMessage(content, sender) {
    const messageDiv = document.createElement('div');
    messageDiv.className = `message ${sender}-message`;
    
    const avatarDiv = document.createElement('div');
    avatarDiv.className = `message-avatar ${sender}-avatar`;
    
    // Add avatar icon
    const avatarIcon = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
    avatarIcon.setAttribute('width', '20');
    avatarIcon.setAttribute('height', '20');
    avatarIcon.setAttribute('viewBox', '0 0 24 24');
    avatarIcon.setAttribute('fill', 'none');
    
    const path = document.createElementNS('http://www.w3.org/2000/svg', 'path');
    path.setAttribute('fill', 'currentColor');
    
    if (sender === 'bot') {
        path.setAttribute('d', 'M12 2C6.48 2 2 6.48 2 12C2 17.52 6.48 22 12 22C17.52 22 22 17.52 22 12C22 6.48 17.52 2 12 2ZM12 5C13.66 5 15 6.34 15 8C15 9.66 13.66 11 12 11C10.34 11 9 9.66 9 8C9 6.34 10.34 5 12 5ZM12 19.2C9.5 19.2 7.29 17.92 6 15.98C6.03 13.99 10 12.9 12 12.9C13.99 12.9 17.97 13.99 18 15.98C16.71 17.92 14.5 19.2 12 19.2Z');
    } else {
        path.setAttribute('d', 'M12 12C14.21 12 16 10.21 16 8C16 5.79 14.21 4 12 4C9.79 4 8 5.79 8 8C8 10.21 9.79 12 12 12ZM12 14C9.33 14 4 15.34 4 18V20H20V18C20 15.34 14.67 14 12 14Z');
    }
    
    avatarIcon.appendChild(path);
    avatarDiv.appendChild(avatarIcon);
    
    const contentDiv = document.createElement('div');
    contentDiv.className = 'message-content';
    
    // Format content (convert markdown-like syntax to HTML)
    contentDiv.innerHTML = formatMessage(content);
    
    messageDiv.appendChild(avatarDiv);
    messageDiv.appendChild(contentDiv);
    
    chatMessages.appendChild(messageDiv);
    scrollToBottom();
}

// Format message content
function formatMessage(content) {
    // Convert line breaks
    let formatted = content.replace(/\n/g, '<br>');
    
    // Convert **bold** to <strong>
    formatted = formatted.replace(/\*\*(.*?)\*\*/g, '<strong>$1</strong>');
    
    // Convert bullet points
    formatted = formatted.replace(/^- (.+)$/gm, '<li>$1</li>');
    formatted = formatted.replace(/(<li>.*<\/li>)/s, '<ul>$1</ul>');
    
    // Convert numbered lists
    formatted = formatted.replace(/^\d+\.\s+(.+)$/gm, '<li>$1</li>');
    
    return formatted;
}

// Show typing indicator
function showTypingIndicator() {
    typingIndicator.style.display = 'flex';
}

// Hide typing indicator
function hideTypingIndicator() {
    typingIndicator.style.display = 'none';
}

// Set processing state
function setProcessingState(processing) {
    isProcessing = processing;
    sendButton.disabled = processing;
    messageInput.disabled = processing;
}

// Show error message
function showError(message) {
    const errorDiv = document.createElement('div');
    errorDiv.className = 'error-message';
    errorDiv.textContent = message;
    
    chatMessages.appendChild(errorDiv);
    scrollToBottom();
    
    // Remove error after 5 seconds
    setTimeout(() => {
        errorDiv.remove();
    }, 5000);
}

// Scroll to bottom of chat
function scrollToBottom() {
    chatMessages.scrollTop = chatMessages.scrollHeight;
}

// Export for embedding
window.DWSChatbot = {
    setApiUrl: (url) => {
        CONFIG.API_URL = url;
    },
    clearHistory: () => {
        conversationHistory = [];
        chatMessages.innerHTML = '';
    }
};
