#!/bin/bash

# DWS Policy Chatbot Deployment Script
# This script helps you deploy the chatbot on a Linux server

set -e  # Exit on error

echo "======================================"
echo "DWS Policy Chatbot Deployment Script"
echo "======================================"
echo ""

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

# Check if running as root
if [ "$EUID" -eq 0 ]; then 
    echo -e "${RED}Please do not run this script as root${NC}"
    exit 1
fi

# Function to print colored output
print_success() {
    echo -e "${GREEN}✓ $1${NC}"
}

print_error() {
    echo -e "${RED}✗ $1${NC}"
}

print_info() {
    echo -e "${YELLOW}ℹ $1${NC}"
}

# Check if Python 3 is installed
echo "Checking prerequisites..."
if ! command -v python3 &> /dev/null; then
    print_error "Python 3 is not installed. Please install Python 3.8 or higher."
    exit 1
fi
print_success "Python 3 is installed"

# Check Python version
PYTHON_VERSION=$(python3 --version | cut -d' ' -f2 | cut -d'.' -f1,2)
print_info "Python version: $PYTHON_VERSION"

# Check if pip is installed
if ! command -v pip3 &> /dev/null; then
    print_error "pip3 is not installed. Installing..."
    sudo apt-get update
    sudo apt-get install -y python3-pip
fi
print_success "pip3 is installed"

# Install backend dependencies
echo ""
echo "Installing backend dependencies..."
cd backend
pip3 install -r requirements.txt --user
print_success "Backend dependencies installed"

# Check for OpenAI API key
echo ""
if [ -z "$OPENAI_API_KEY" ]; then
    print_error "OPENAI_API_KEY environment variable is not set"
    echo ""
    echo "Please set your OpenAI API key:"
    echo "  export OPENAI_API_KEY='your-api-key-here'"
    echo ""
    echo "Add this to your ~/.bashrc or ~/.profile to make it permanent:"
    echo "  echo 'export OPENAI_API_KEY=\"your-api-key-here\"' >> ~/.bashrc"
    echo "  source ~/.bashrc"
    echo ""
    exit 1
else
    print_success "OPENAI_API_KEY is set"
fi

# Ask if user wants to start the server now
echo ""
read -p "Do you want to start the backend server now? (y/n) " -n 1 -r
echo ""
if [[ $REPLY =~ ^[Yy]$ ]]; then
    print_info "Starting backend server on http://localhost:8000"
    print_info "Press Ctrl+C to stop the server"
    echo ""
    python3 app.py
else
    echo ""
    print_info "To start the backend server manually, run:"
    echo "  cd backend"
    echo "  python3 app.py"
    echo ""
    print_info "To start the frontend, open frontend/index.html in a browser"
    echo "  Or serve it with: cd frontend && python3 -m http.server 8080"
fi

echo ""
print_success "Deployment completed!"
