"""
Company Policy Chatbot Backend - Google Docs Integration
A FastAPI application that fetches policies from Google Docs and provides 
intelligent responses using OpenAI GPT models.
"""

from fastapi import FastAPI, HTTPException
from fastapi.middleware.cors import CORSMiddleware
from pydantic import BaseModel
from openai import OpenAI
import os
import requests
from typing import List
from datetime import datetime
import httpx

# Initialize FastAPI app
app = FastAPI(
    title="DWS Company Policy Chatbot API",
    description="Intelligent chatbot with Google Docs integration for company policies",
    version="2.1.0"
)

# Configure CORS
app.add_middleware(
    CORSMiddleware,
    allow_origins=["*"],
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)

# Configuration
GOOGLE_DOC_ID = os.environ.get("GOOGLE_DOC_ID", "1I3XoFUvTFgjJPck8MahPeAIMuodlkPZSj4Y7kyVXscI")
GOOGLE_DOC_URL = f"https://docs.google.com/document/d/{GOOGLE_DOC_ID}/export?format=txt"

# Initialize OpenAI client
api_key = os.environ.get("OPENAI_API_KEY")
if not api_key:
    raise Exception("OPENAI_API_KEY environment variable is not set")

# Create httpx client manually to avoid proxies issue with Python 3.13
http_client = httpx.Client(
    timeout=30.0,
    follow_redirects=True
)

client = OpenAI(
    api_key=api_key,
    http_client=http_client
)

# Global variables
COMPANY_POLICIES = ""
LAST_FETCH_TIME = None
POLICY_SOURCE = "unknown"

def load_policies_from_google_docs():
    """Load company policies from Google Docs"""
    global LAST_FETCH_TIME, POLICY_SOURCE
    try:
        print(f"📥 Fetching policies from Google Docs...")
        print(f"   URL: {GOOGLE_DOC_URL}")
        
        response = requests.get(GOOGLE_DOC_URL, timeout=15)
        response.raise_for_status()
        
        content = response.text.strip()
        
        if len(content) < 100:
            raise Exception("Document content too short - may not be publicly accessible")
        
        LAST_FETCH_TIME = datetime.now()
        POLICY_SOURCE = "Google Docs"
        
        print(f"✅ Successfully loaded {len(content)} characters from Google Docs")
        print(f"   First 100 chars: {content[:100]}...")
        
        return content
        
    except Exception as e:
        print(f"❌ Error loading from Google Docs: {str(e)}")
        raise

def load_policies_from_file():
    """Load company policies from local file (fallback)"""
    global POLICY_SOURCE
    policy_file = os.path.join(os.path.dirname(__file__), "..", "data", "company_policies.txt")
    try:
        with open(policy_file, "r", encoding="utf-8") as f:
            content = f.read()
        POLICY_SOURCE = "Local File"
        print(f"✅ Loaded {len(content)} characters from local file")
        return content
    except Exception as e:
        print(f"❌ Error loading from file: {str(e)}")
        raise

# Initialize policies - try Google Docs first, fallback to local file
print("=" * 60)
print("🚀 Initializing DWS Company Policy Chatbot")
print("=" * 60)

try:
    COMPANY_POLICIES = load_policies_from_google_docs()
    print(f"📄 Policy Source: Google Docs")
except Exception as e:
    print(f"⚠️  Google Docs unavailable: {str(e)}")
    print(f"📂 Attempting to load from local file...")
    try:
        COMPANY_POLICIES = load_policies_from_file()
        print(f"📄 Policy Source: Local File (fallback)")
    except Exception as e2:
        print(f"❌ Failed to load from local file: {str(e2)}")
        COMPANY_POLICIES = ""
        POLICY_SOURCE = "None"

if COMPANY_POLICIES:
    print(f"✅ Policies loaded successfully")
    print(f"   Length: {len(COMPANY_POLICIES)} characters")
    print(f"   Lines: {len(COMPANY_POLICIES.splitlines())}")
else:
    print(f"⚠️  WARNING: No policies loaded!")

print("=" * 60)

# Request/Response Models
class ChatMessage(BaseModel):
    role: str
    content: str

class ChatRequest(BaseModel):
    message: str
    conversation_history: List[ChatMessage] = []

# System Prompt
SYSTEM_PROMPT = """You are Maya, a friendly and professional HR Policy Assistant for Digital Web Solutions (DWS) and its group of companies. You help employees understand and navigate company policies with warmth, clarity, and professionalism.

🎯 YOUR CORE MISSION:
Help employees get accurate, helpful information about company policies while maintaining a professional yet approachable demeanor.

📋 RESPONSE GUIDELINES:

1. **GREETINGS & SMALL TALK** (Be Warm & Welcoming):
   When users say: "Hello", "Hi", "Hey", "Good morning", "Good afternoon", "Good evening", "How are you", etc.
   
   Respond warmly:
   "Hello! 👋 I'm your DWS Policy Assistant. I'm here to help you understand our company policies and answer any questions you might have.
   
   I can assist you with:
   • Leave policies (sick leave, privilege leave, maternity/paternity leave)
   • Resignation procedures and notice periods
   • Appointment and probation information
   • Employee code of conduct
   • Working hours and attendance policies
   • Contact information for HR, IT, Admin, and Finance teams
   • And much more!
   
   What would you like to know today?"

2. **POLICY QUESTIONS** (Answer Accurately):
   ✅ Answer ONLY from the company policy handbook provided below
   ✅ Be specific and cite the relevant policy section
   ✅ Use clear formatting with bullet points or numbered lists
   ✅ Be concise but complete
   ✅ If information is not in the handbook, be honest and direct to HR
   
   Example format:
   "According to the **Leave Policy** in our handbook:
   
   • Sick Leave: 7 days per year (accrued at 3.5 days every 6 months)
   • Privilege Leave: 15 days per year
   • Unused sick leave expires at year-end
   
   Would you like to know more about any specific leave type?"

3. **OFF-TOPIC QUESTIONS** (Politely Redirect):
   If asked about: weather, news, general knowledge, entertainment, math, technical troubleshooting, or anything NOT in the policy handbook:
   
   "Thanks for your question! I’m here to help with DWS company policies — I don’t have info on the [topic mentioned]."

4. **UNCLEAR OR VAGUE QUESTIONS** (Clarify Helpfully):
   If the question is vague, ask for clarification while offering suggestions:
   
   "I'd be happy to help! Could you be more specific? For example, are you asking about:
   • Leave policies?
   • Resignation procedures?
   • Contact information?
   • Something else?
   
   Let me know and I'll provide the exact information you need!"

5. **MISSING INFORMATION** (Be Honest & Helpful):
   If the policy handbook doesn't contain the answer:
   
   "I couldn’t find any mention of an 'unlimited leave' option in our current policies. Typically, leaves are defined with specific limits for each category.

For the most accurate and personalized details, please contact:
📧 HR: hr@dwsmail.com

📞 Or reach out to your HR Partner directly.

Would you like me to share the general leave categories we offer?"

6. **FORMATTING BEST PRACTICES**:
   • Use bullet points (•) for lists
   • Use **bold** for policy names and important terms
   • Keep paragraphs short (2-3 sentences max)
   • Use emojis sparingly and professionally (✅, 📧, 📞, etc.)
   • End with a helpful follow-up question

7. **TONE & PERSONALITY**:
   • Professional yet friendly
   • Patient and understanding
   • Clear and concise
   • Empathetic (especially for sensitive topics like resignation, leave)
   • Proactive in offering additional help

🔒 SECURITY RULES:
- NEVER make up policy information
- NEVER answer questions outside the policy handbook (except greetings)
- NEVER provide personal advice beyond company policies
- ALWAYS cite the source policy when answering

📚 COMPANY POLICY HANDBOOK:
{policies}

---

Remember: You're representing DWS. Be helpful, accurate, and professional. Every interaction should leave employees feeling informed and supported.
"""

@app.get("/")
async def root():
    """Root endpoint"""
    return {
        "status": "online",
        "service": "DWS Company Policy Chatbot API",
        "version": "2.1.0",
        "policy_source": POLICY_SOURCE,
        "last_updated": LAST_FETCH_TIME.isoformat() if LAST_FETCH_TIME else "N/A",
        "timestamp": datetime.now().isoformat()
    }

@app.get("/health")
async def health_check():
    """Health check endpoint"""
    return {
        "status": "healthy",
        "policies_loaded": len(COMPANY_POLICIES) > 0,
        "policy_length": len(COMPANY_POLICIES),
        "policy_lines": len(COMPANY_POLICIES.splitlines()),
        "policy_source": POLICY_SOURCE,
        "google_doc_id": GOOGLE_DOC_ID,
        "last_updated": LAST_FETCH_TIME.isoformat() if LAST_FETCH_TIME else "N/A"
    }

@app.post("/chat")
async def chat(request: ChatRequest):
    """Process chat message and return response"""
    try:
        # Build messages
        messages = [
            {"role": "system", "content": SYSTEM_PROMPT.format(policies=COMPANY_POLICIES)}
        ]
        
        # Add conversation history (last 10 messages)
        for msg in request.conversation_history[-10:]:
            messages.append({"role": msg.role, "content": msg.content})
        
        # Add current message
        messages.append({"role": "user", "content": request.message})
        
        # Call OpenAI
        response = client.chat.completions.create(
            model="gpt-4o-mini",
            messages=messages,
            temperature=0.7,
            max_tokens=1200,
            frequency_penalty=0.3,
            presence_penalty=0.2
        )
        
        return {
            "response": response.choices[0].message.content,
            "timestamp": datetime.now().isoformat()
        }
        
    except Exception as e:
        print(f"❌ Error in chat: {str(e)}")
        raise HTTPException(status_code=500, detail=str(e))

@app.post("/reload-policies")
async def reload_policies():
    """Reload policies from Google Docs"""
    global COMPANY_POLICIES
    
    try:
        print("🔄 Reloading policies from Google Docs...")
        COMPANY_POLICIES = load_policies_from_google_docs()
        
        return {
            "status": "success",
            "message": "Policies reloaded successfully from Google Docs",
            "policy_length": len(COMPANY_POLICIES),
            "policy_source": POLICY_SOURCE,
            "timestamp": datetime.now().isoformat()
        }
        
    except Exception as e:
        print(f"⚠️  Google Docs reload failed, trying local file...")
        try:
            COMPANY_POLICIES = load_policies_from_file()
            return {
                "status": "partial_success",
                "message": f"Google Docs unavailable, loaded from local file: {str(e)}",
                "policy_length": len(COMPANY_POLICIES),
                "policy_source": POLICY_SOURCE,
                "timestamp": datetime.now().isoformat()
            }
        except Exception as e2:
            raise HTTPException(
                status_code=500,
                detail=f"Failed to reload from both sources. Google Docs: {str(e)}, Local: {str(e2)}"
            )

@app.get("/policy-info")
async def policy_info():
    """Get information about current policies"""
    return {
        "google_doc_id": GOOGLE_DOC_ID,
        "google_doc_url": GOOGLE_DOC_URL,
        "policy_source": POLICY_SOURCE,
        "policy_length": len(COMPANY_POLICIES),
        "policy_lines": len(COMPANY_POLICIES.splitlines()),
        "last_updated": LAST_FETCH_TIME.isoformat() if LAST_FETCH_TIME else "N/A",
        "preview": COMPANY_POLICIES[:300] + "..." if len(COMPANY_POLICIES) > 300 else COMPANY_POLICIES
    }

if __name__ == "__main__":
    import uvicorn
    uvicorn.run(app, host="0.0.0.0", port=8000)
