# DWS Company Policy Chatbot

A professional, AI-powered chatbot for answering employee queries about company policies. Built with FastAPI backend and modern web frontend, easily embeddable in any website.

## 🌟 Features

- **Intelligent Responses**: Uses OpenAI GPT models to provide accurate, context-aware answers
- **Easy Content Management**: Update policies by simply editing a text file - no coding required
- **Embeddable Widget**: Can be embedded in any website with a simple iframe or widget
- **Conversation History**: Maintains context across multiple questions
- **Professional UI**: Clean, modern interface with responsive design
- **Production Ready**: Built with FastAPI for high performance and scalability

## 📁 Project Structure

```
company-policy-chatbot/
├── backend/
│   ├── app.py              # Main FastAPI application
│   └── requirements.txt    # Python dependencies
├── frontend/
│   ├── index.html          # Main chat interface
│   ├── style.css           # Styling
│   ├── script.js           # Chat functionality
│   ├── embed.html          # Embeddable widget version
│   └── admin.html          # Content management interface
├── data/
│   └── company_policies.txt # Your company policies (editable)
└── README.md               # This file
```

## 🚀 Quick Start

### Prerequisites

- Python 3.8 or higher
- OpenAI API key (or compatible API key)
- Basic knowledge of command line

### Installation

1. **Navigate to the project directory**
   ```bash
   cd company-policy-chatbot
   ```

2. **Install Python dependencies**
   ```bash
   cd backend
   pip3 install -r requirements.txt
   ```

3. **Set up your OpenAI API key**
   
   **On Linux/Mac:**
   ```bash
   export OPENAI_API_KEY="your-api-key-here"
   ```
   
   **On Windows (Command Prompt):**
   ```cmd
   set OPENAI_API_KEY=your-api-key-here
   ```
   
   **On Windows (PowerShell):**
   ```powershell
   $env:OPENAI_API_KEY="your-api-key-here"
   ```

4. **Start the backend server**
   ```bash
   python3 app.py
   ```
   
   The server will start on `http://localhost:8000`

5. **Open the frontend**
   
   Open `frontend/index.html` in your web browser, or serve it using a simple HTTP server:
   
   ```bash
   cd ../frontend
   python3 -m http.server 8080
   ```
   
   Then visit `http://localhost:8080`

## 📝 Updating Company Policies

### Method 1: Edit the Text File (Recommended)

1. Navigate to the `data` folder
2. Open `company_policies.txt` in any text editor
3. Make your changes (add, edit, or remove content)
4. Save the file
5. Reload policies:
   - Visit `http://localhost:8080/admin.html` and click "Reload Policies"
   - OR use the API: `curl -X POST http://localhost:8000/reload-policies`

### Method 2: Replace the Entire File

1. Prepare your new policy document as a plain text file
2. Replace `data/company_policies.txt` with your new file
3. Keep the filename as `company_policies.txt`
4. Reload policies using the admin panel or API

**Important**: No server restart is required! Just reload the policies.

## 🌐 Embedding in Your Website

### Option 1: Full Page Embed

Add this to your website's HTML:

```html
<iframe 
    src="http://your-server:8080/index.html" 
    width="100%" 
    height="600px" 
    frameborder="0"
    title="Policy Assistant">
</iframe>
```

### Option 2: Widget (Floating Button)

Add this to your website's HTML:

```html
<iframe 
    src="http://your-server:8080/embed.html" 
    style="position: fixed; bottom: 0; right: 0; width: 100%; height: 100%; border: none; z-index: 9999; pointer-events: none;"
    title="Policy Assistant Widget">
</iframe>
```

The widget will appear as a floating button in the bottom-right corner.

### Option 3: Custom Integration

Update the API URL in `frontend/script.js`:

```javascript
const CONFIG = {
    API_URL: 'http://your-server-domain:8000',  // Change this
    MAX_RETRIES: 3,
    RETRY_DELAY: 1000
};
```

## 🖥️ Deploying to Production

### 1. Prepare Your Server

- Ubuntu/Debian Linux server (recommended)
- Python 3.8+
- Nginx (for serving frontend and reverse proxy)
- Supervisor or systemd (for keeping backend running)

### 2. Deploy Backend

1. **Copy the project to your server**
   ```bash
   scp -r company-policy-chatbot user@your-server:/var/www/
   ```

2. **Install dependencies**
   ```bash
   cd /var/www/company-policy-chatbot/backend
   pip3 install -r requirements.txt
   ```

3. **Create a systemd service** (create file: `/etc/systemd/system/policy-chatbot.service`)
   ```ini
   [Unit]
   Description=DWS Policy Chatbot Backend
   After=network.target

   [Service]
   Type=simple
   User=www-data
   WorkingDirectory=/var/www/company-policy-chatbot/backend
   Environment="OPENAI_API_KEY=your-api-key-here"
   ExecStart=/usr/bin/python3 /var/www/company-policy-chatbot/backend/app.py
   Restart=always

   [Install]
   WantedBy=multi-user.target
   ```

4. **Start the service**
   ```bash
   sudo systemctl daemon-reload
   sudo systemctl start policy-chatbot
   sudo systemctl enable policy-chatbot
   ```

### 3. Configure Nginx

Create nginx configuration (file: `/etc/nginx/sites-available/policy-chatbot`):

```nginx
server {
    listen 80;
    server_name your-domain.com;

    # Frontend
    location / {
        root /var/www/company-policy-chatbot/frontend;
        index index.html;
        try_files $uri $uri/ =404;
    }

    # Backend API
    location /api/ {
        proxy_pass http://localhost:8000/;
        proxy_http_version 1.1;
        proxy_set_header Upgrade $http_upgrade;
        proxy_set_header Connection 'upgrade';
        proxy_set_header Host $host;
        proxy_cache_bypass $http_upgrade;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
    }
}
```

Enable the site:
```bash
sudo ln -s /etc/nginx/sites-available/policy-chatbot /etc/nginx/sites-enabled/
sudo nginx -t
sudo systemctl restart nginx
```

### 4. Update Frontend Configuration

Edit `frontend/script.js` and update the API URL:

```javascript
const CONFIG = {
    API_URL: 'http://your-domain.com/api',  // Update this
    MAX_RETRIES: 3,
    RETRY_DELAY: 1000
};
```

### 5. Set Up SSL (Recommended)

```bash
sudo apt install certbot python3-certbot-nginx
sudo certbot --nginx -d your-domain.com
```

## 🔧 Configuration Options

### Backend Configuration

Edit `backend/app.py` to customize:

- **Model Selection**: Change `model="gpt-4.1-mini"` to use different models
- **Temperature**: Adjust `temperature=0.7` (0.0 = deterministic, 1.0 = creative)
- **Max Tokens**: Change `max_tokens=1000` to control response length
- **CORS Origins**: Update `allow_origins=["*"]` to restrict access

### Frontend Configuration

Edit `frontend/script.js`:

- **API URL**: Update `CONFIG.API_URL`
- **Retry Settings**: Adjust `MAX_RETRIES` and `RETRY_DELAY`

## 📊 API Documentation

Once the backend is running, visit:
- API Docs: `http://localhost:8000/docs`
- Alternative Docs: `http://localhost:8000/redoc`

### Main Endpoints

- `GET /` - Health check
- `GET /health` - Detailed health status
- `POST /chat` - Send a message and get response
- `POST /reload-policies` - Reload policy content from file
- `GET /policy-info` - Get information about loaded policies

## 🧪 Testing

### Test Backend

```bash
# Check health
curl http://localhost:8000/health

# Test chat
curl -X POST http://localhost:8000/chat \
  -H "Content-Type: application/json" \
  -d '{"message": "What is the leave policy?"}'
```

### Test Frontend

1. Open `http://localhost:8080/index.html`
2. Ask a question like "What is the sick leave policy?"
3. Verify the response is accurate

## 🔒 Security Considerations

1. **API Key**: Never commit your OpenAI API key to version control
2. **CORS**: In production, restrict CORS to your specific domain
3. **Rate Limiting**: Consider adding rate limiting to prevent abuse
4. **Authentication**: Add authentication if the chatbot contains sensitive information
5. **HTTPS**: Always use HTTPS in production

## 🐛 Troubleshooting

### Backend won't start
- Check if port 8000 is already in use
- Verify OpenAI API key is set correctly
- Check Python version (must be 3.8+)

### Frontend can't connect to backend
- Verify backend is running: `curl http://localhost:8000/health`
- Check API URL in `script.js` matches your backend
- Check browser console for CORS errors

### Chatbot gives incorrect answers
- Verify policy file is loaded: `curl http://localhost:8000/policy-info`
- Check if policy content is properly formatted
- Try reloading policies: `curl -X POST http://localhost:8000/reload-policies`

### "Cannot connect to backend server" error
- Ensure backend is running
- Check firewall settings
- Verify the API URL is correct

## 📞 Support

For questions or issues:
- Check the troubleshooting section above
- Review API documentation at `/docs`
- Contact your IT team for deployment assistance

## 📄 License

This project is proprietary software for Digital Web Solutions (DWS) and its group of companies.

## 🎯 Next Steps

1. Test the chatbot thoroughly with various questions
2. Update the company policies in `data/company_policies.txt`
3. Customize the UI colors and branding in `style.css`
4. Deploy to your production server
5. Embed in your company website
6. Train employees on how to use the chatbot

---

**Version**: 1.0.0  
**Last Updated**: October 2025  
**Developed for**: Digital Web Solutions (DWS)
