# Deployment Guide for Development Team

## 📦 What to Deploy

**Give your dev team the ENTIRE ZIP file.** All files are needed:

```
company-policy-chatbot/
├── backend/              # Python FastAPI server
│   ├── app.py           # Main application (REQUIRED)
│   └── requirements.txt # Python dependencies (REQUIRED)
├── data/
│   └── company_policies.txt  # Policy content (REQUIRED)
├── frontend/            # Web interface
│   ├── index.html      # Main chat page (REQUIRED)
│   ├── style.css       # Styling (REQUIRED)
│   ├── script.js       # Chat logic (REQUIRED)
│   ├── admin.html      # Admin panel (REQUIRED)
│   └── embed.html      # Widget demo (OPTIONAL)
├── start.sh            # Linux startup script
├── nginx.conf          # Nginx configuration example
├── policy-chatbot.service  # Systemd service example
└── README.md           # Documentation
```

---

## 🚀 Deployment Steps for Dev Team

### Step 1: Server Requirements

**Minimum Server Specs:**
- OS: Ubuntu 20.04+ or any Linux distribution
- RAM: 512MB minimum (1GB recommended)
- CPU: 1 core minimum
- Storage: 500MB
- Python: 3.8 or higher

**Required Software:**
```bash
sudo apt update
sudo apt install -y python3 python3-pip nginx
```

---

### Step 2: Upload Files to Server

Upload the entire `company-policy-chatbot` folder to the server, for example:
```
/var/www/company-policy-chatbot/
```

---

### Step 3: Install Python Dependencies

```bash
cd /var/www/company-policy-chatbot/backend
pip3 install -r requirements.txt
```

---

### Step 4: Set Environment Variables

Create a `.env` file or set environment variable:

```bash
export OPENAI_API_KEY="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"
```

Or create `/var/www/company-policy-chatbot/.env`:
```
OPENAI_API_KEY=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
```

---

### Step 5: Set Up Systemd Service (Auto-start on boot)

Copy the provided `policy-chatbot.service` file:

```bash
sudo cp /var/www/company-policy-chatbot/policy-chatbot.service /etc/systemd/system/
```

Edit the file to match your paths:
```bash
sudo nano /etc/systemd/system/policy-chatbot.service
```

Enable and start the service:
```bash
sudo systemctl daemon-reload
sudo systemctl enable policy-chatbot
sudo systemctl start policy-chatbot
sudo systemctl status policy-chatbot
```

The backend will now run on `http://localhost:8000`

---

### Step 6: Configure Nginx (Web Server)

Copy the provided nginx configuration:

```bash
sudo cp /var/www/company-policy-chatbot/nginx.conf /etc/nginx/sites-available/policy-chatbot
```

Edit the file to set your domain:
```bash
sudo nano /etc/nginx/sites-available/policy-chatbot
```

Change `your-domain.com` to your actual domain (e.g., `chatbot.yourcompany.com`)

Enable the site:
```bash
sudo ln -s /etc/nginx/sites-available/policy-chatbot /etc/nginx/sites-enabled/
sudo nginx -t  # Test configuration
sudo systemctl reload nginx
```

---

### Step 7: Set Up SSL Certificate (HTTPS)

Install Certbot:
```bash
sudo apt install -y certbot python3-certbot-nginx
```

Get SSL certificate:
```bash
sudo certbot --nginx -d chatbot.yourcompany.com
```

Follow the prompts. Certbot will automatically configure HTTPS.

---

### Step 8: Test the Deployment

1. **Test backend:**
   ```bash
   curl http://localhost:8000/health
   ```
   Should return: `{"status":"healthy","policies_loaded":true}`

2. **Test frontend:**
   Open browser to: `https://chatbot.yourcompany.com`

3. **Test chat:**
   Ask a question in the chat interface

---

## 🌐 Embed Code for Your Website

After deployment, your dev team will provide you with the chatbot URL. There are **3 ways** to embed it:

### Option 1: Floating Chat Widget (Recommended)

Add this code to your website **before the closing `</body>` tag**:

```html
<!-- DWS Policy Chatbot Widget -->
<script>
(function() {
    // Create button
    var btn = document.createElement('button');
    btn.innerHTML = '💬';
    btn.style.cssText = 'position:fixed;bottom:20px;right:20px;width:60px;height:60px;border-radius:50%;background:#292F4C;color:white;border:none;font-size:28px;cursor:pointer;box-shadow:0 4px 12px rgba(0,0,0,0.3);z-index:9998;transition:transform 0.2s;';
    btn.onmouseover = function() { this.style.transform = 'scale(1.1)'; };
    btn.onmouseout = function() { this.style.transform = 'scale(1)'; };
    
    // Create iframe container
    var container = document.createElement('div');
    container.style.cssText = 'position:fixed;bottom:90px;right:20px;width:400px;height:600px;border:none;border-radius:16px;box-shadow:0 8px 24px rgba(0,0,0,0.3);z-index:9999;display:none;';
    
    var iframe = document.createElement('iframe');
    iframe.src = 'https://chatbot.yourcompany.com/index.html';  // REPLACE WITH YOUR DOMAIN
    iframe.style.cssText = 'width:100%;height:100%;border:none;border-radius:16px;';
    
    container.appendChild(iframe);
    
    // Toggle chat
    btn.onclick = function() {
        if (container.style.display === 'none') {
            container.style.display = 'block';
        } else {
            container.style.display = 'none';
        }
    };
    
    document.body.appendChild(btn);
    document.body.appendChild(container);
})();
</script>
```

**Result:** A floating chat button in the bottom-right corner. Click to open/close the chatbot.

---

### Option 2: Embedded in Page Section

Add this code where you want the chatbot to appear on your page:

```html
<!-- DWS Policy Chatbot - Embedded -->
<div style="width: 100%; max-width: 800px; height: 600px; margin: 0 auto;">
    <iframe 
        src="https://chatbot.yourcompany.com/index.html" 
        style="width: 100%; height: 100%; border: none; border-radius: 16px; box-shadow: 0 4px 12px rgba(0,0,0,0.1);"
        title="Company Policy Chatbot">
    </iframe>
</div>
```

**Result:** Chatbot embedded directly in your page content.

---

### Option 3: Full Page

Create a dedicated page for the chatbot:

```html
<!DOCTYPE html>
<html>
<head>
    <title>Company Policy Assistant</title>
    <style>
        body { margin: 0; padding: 0; overflow: hidden; }
        iframe { width: 100vw; height: 100vh; border: none; }
    </style>
</head>
<body>
    <iframe src="https://chatbot.yourcompany.com/index.html"></iframe>
</body>
</html>
```

**Result:** Full-page chatbot experience.

---

## 🔧 Configuration for Dev Team

### Important Files to Configure:

1. **backend/app.py** - Already configured, no changes needed
2. **nginx.conf** - Change `your-domain.com` to actual domain
3. **policy-chatbot.service** - Change paths if not using `/var/www/`
4. **.env** - Add the OpenAI API key

### Ports Used:
- **Backend (API):** Port 8000
- **Frontend:** Served by Nginx on port 80/443

### CORS Configuration:
The backend is already configured to allow all origins (`allow_origins=["*"]`). For production, your dev team can restrict this to specific domains:

```python
# In backend/app.py, line 24-30
app.add_middleware(
    CORSMiddleware,
    allow_origins=["https://yourcompany.com", "https://www.yourcompany.com"],  # Specific domains
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)
```

---

## 📝 After Deployment Checklist

Your dev team should verify:

- [ ] Backend running on port 8000
- [ ] Health check returns success: `curl http://localhost:8000/health`
- [ ] Frontend accessible via domain
- [ ] SSL certificate installed (HTTPS)
- [ ] Chat responds to questions
- [ ] Admin panel works: `https://chatbot.yourcompany.com/admin.html`
- [ ] Policies loaded correctly
- [ ] CORS allows your main website

---

## 🎯 What Dev Team Will Give You

After deployment, your dev team will provide:

1. **Chatbot URL:** `https://chatbot.yourcompany.com`
2. **Embed code** (one of the 3 options above with correct URL)
3. **Admin panel URL:** `https://chatbot.yourcompany.com/admin.html`
4. **API health check:** `https://chatbot.yourcompany.com/api/health`

---

## 🔄 Updating Policies (For You - No Dev Team Needed)

After deployment, YOU can update policies yourself:

### Method 1: Via Server (Ask dev team to give you access)
1. SSH to server
2. Edit file: `nano /var/www/company-policy-chatbot/data/company_policies.txt`
3. Save changes
4. Go to admin panel and click "Reload Policies"

### Method 2: Via Admin Panel (If file upload is added)
1. Edit policies in Notepad on your computer
2. Upload via admin panel
3. Click "Reload Policies"

**No restart needed!** Changes take effect immediately.

---

## 💰 Estimated Costs

**Server Hosting:**
- Basic VPS: $5-10/month (DigitalOcean, Linode, Vultr)
- Or use existing company server (free)

**OpenAI API:**
- Model: gpt-4o-mini (most cost-effective)
- Estimated: $5-25/month for 100 employees
- Depends on usage volume

**Domain & SSL:**
- Subdomain: Free (use existing domain)
- SSL Certificate: Free (Let's Encrypt)

**Total: ~$10-35/month**

---

## 🆘 Troubleshooting

### Backend not starting:
```bash
sudo systemctl status policy-chatbot
sudo journalctl -u policy-chatbot -n 50
```

### Frontend not loading:
```bash
sudo nginx -t
sudo systemctl status nginx
```

### Chat not responding:
- Check backend health: `curl http://localhost:8000/health`
- Check API key is set: `echo $OPENAI_API_KEY`
- Check logs: `sudo journalctl -u policy-chatbot -f`

---

## 📞 Support

If your dev team has questions:
1. Check `README.md` for detailed documentation
2. Check `DEPLOYMENT_CHECKLIST.md` for step-by-step guide
3. All configuration files are included with comments

---

## ✅ Summary for Dev Team

**What to do:**
1. Upload entire folder to server
2. Install Python dependencies
3. Set OpenAI API key as environment variable
4. Configure systemd service
5. Configure nginx with your domain
6. Get SSL certificate
7. Test deployment
8. Provide embed code with correct URL

**Time estimate:** 30-60 minutes for experienced dev

**Difficulty:** Easy (standard Python + Nginx deployment)
